/******************************************************************************
 Copyright (C) 2014 by John R. Bradley <jrb@turrettech.com>
 Copyright (C) 2018 by Hugh Bailey ("Jim") <jim@obsproject.com>

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/


#include <obs-frontend-api.h>
#include <util/threading.h>
#include <util/platform.h>
#include <util/util.hpp>
#include <util/dstr.hpp>
#include <obs-module.h>
#include <obs.hpp>
#include <thread>
#include <mutex>


#include "obs-browser-source.hpp"
#include "browser-scheme.hpp"
#include "browser-app.hpp"
#include "browser-version.h"
#include "browser-config.h"

#include "deps/json11/json11.hpp"
#include "cef-headers.hpp"

#include <process.h>
#include <string>
#include "./zip/unzip.h"
#include "./zip/zip.h"
#include "libcurl/inc/CHttpClient.h"
#include <iostream>
#include <list>
#include <shlwapi.h>

#ifdef _WIN32
#include <util/windows/ComPtr.hpp>
#include <dxgi.h>
#include <dxgi1_2.h>
#include <d3d11.h>
#endif

#include "server/websocket_server/ws_examples.h"
#include "libcurl/inc/curl_examples.h"
#pragma comment(lib,"shlwapi.lib")



OBS_DECLARE_MODULE()
OBS_MODULE_USE_DEFAULT_LOCALE("Lovense-Video-Feedback", "en-US")

using namespace std;
using namespace json11;

static thread manager_thread;
os_event_t *cef_started_event = nullptr;

static int adapterCount = 0;
static std::wstring deviceId;

#if EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
bool hwaccel = false;
#endif

//TODO: Add plugins to sense
void add_sense_to_obs();
/* ========================================================================= */

class BrowserTask : public CefTask {
public:
	std::function<void()> task;

	inline BrowserTask(std::function<void()> task_) : task(task_) {}
	virtual void Execute() override { task(); }

	IMPLEMENT_REFCOUNTING(BrowserTask);
};

bool QueueCEFTask(std::function<void()> task)
{
	return CefPostTask(TID_UI, CefRefPtr<BrowserTask>(new BrowserTask(task)));
}

/* ========================================================================= */

static const char *default_css = "\
body { \
background-color: rgba(0, 0, 0, 0); \
margin: 0px auto; \
overflow: hidden; \
}";

static void ReplaceChar(char* pStr, char cDest, char cSrc)
{
	char* pTmp = pStr;
	while ('\0' != *pTmp)
	{
		if (*pTmp == cSrc)
			*pTmp = cDest;
		pTmp++;
	}
}

static void browser_source_get_defaults(obs_data_t *settings)
{
	char path[512] = { 0 };
	uint64_t cx = 800;
	uint64_t cy = 600;
	if (os_get_config_path(path, sizeof(path), "obs-studio") <= 0)
		blog(LOG_ERROR, "[Lovese] os_get_config_path locale dist path is  fail ");
	else
	{
		ConfigFile  globalConfig;
		std::string configPath = path;
		std::string globalFile = configPath + "/global.ini";

		if (globalConfig.Open(globalFile.c_str(), CONFIG_OPEN_ALWAYS) == CONFIG_SUCCESS)
		{
			const char *ProfileDir = config_get_string(globalConfig, "Basic",
				"ProfileDir");
			if (!ProfileDir)
				ProfileDir = config_get_string(globalConfig, "Basic",
					"SceneCollection");
			if (ProfileDir)
			{
				blog(LOG_INFO, "ProfileDIr1 %s", ProfileDir);

				std::string profile = configPath + "/basic/profiles/" + ProfileDir + "/basic.ini";
				ConfigFile basicConfig;
				if (basicConfig.Open(profile.c_str(), CONFIG_OPEN_EXISTING) == CONFIG_SUCCESS)
				{
					cx = config_get_uint(basicConfig, "Video", "BaseCX");
					cy = config_get_uint(basicConfig, "Video", "BaseCY");
					blog(LOG_INFO, "ProfileDIr2 %s", ProfileDir);

				}
				blog(LOG_INFO, "ProfileDIr3 %s", ProfileDir);
			}
		}
	}
	

	obs_data_set_default_int(settings, "width", cx);
	obs_data_set_default_int(settings, "height", cy);
	obs_data_set_default_int(settings, "fps", 30);
#if EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
	obs_data_set_default_bool(settings, "fps_custom", false);
#else
	obs_data_set_bool(settings, "fps_custom", true);
#endif
	obs_data_set_default_bool(settings, "shutdown", false);
	obs_data_set_default_bool(settings, "restart_when_active", false);
	obs_data_set_default_string(settings, "css", default_css);

	obs_data_set_default_bool(settings, "is_local_file", true);


	char szLocalFile[512] = { 0 };
	if (os_get_config_path(szLocalFile, sizeof(szLocalFile), "obs-studio\\dist\\index.html") <= 0)
		return;

	char szProPath[1024] = { 0 };
	sprintf_s(szProPath, "file://%s?port=81000", szLocalFile);

	ReplaceChar(szProPath, '/', '\\');

	obs_data_set_default_bool(settings, "is_local_file", false);
	obs_data_set_default_string(settings, "url", szProPath);
}

static bool is_local_file_modified(obs_properties_t *props,
	obs_property_t *, obs_data_t *settings)
{
	bool enabled = obs_data_get_bool(settings, "is_local_file");
	obs_property_t *url = obs_properties_get(props, "url");
	obs_property_t *local_file = obs_properties_get(props, "local_file");
	obs_property_set_visible(url, !enabled);
	obs_property_set_visible(local_file, enabled);

	return true;
}

static bool is_fps_custom(obs_properties_t *props,
	obs_property_t *, obs_data_t *settings)
{
	bool enabled = obs_data_get_bool(settings, "fps_custom");
	obs_property_t *fps = obs_properties_get(props, "fps");
	obs_property_set_visible(fps, enabled);

	return true;
}

bool pexls_list_property_modified_t(obs_properties_t *props, obs_property_t *property, obs_data_t *settings)
{
	if (obs_property_get_type(property) == OBS_PROPERTY_LIST)
	{
		if (obs_property_list_format(property) == OBS_COMBO_FORMAT_STRING)
		{
		     const char * json = obs_data_get_string(settings,"Pix");
		     printf("");
		}
	}
	return true;
}


static obs_properties_t *browser_source_get_properties(void *data)
{
	obs_properties_t *props = obs_properties_create();
	BrowserSource *bs = static_cast<BrowserSource *>(data);
	DStr path;

	obs_properties_set_flags(props, OBS_PROPERTIES_DEFER_UPDATE);

	if (bs && !bs->url.empty()) {
		const char *slash;

		dstr_copy(path, bs->url.c_str());
		dstr_replace(path, "\\", "/");
		slash = strrchr(path->array, '/');
		if (slash)
			dstr_resize(path, slash - path->array + 1);
	}


#if !EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
	
#endif

	
	return props;
}

static void BrowserManagerThread(void)
{
	string path = obs_get_module_binary_path(obs_current_module());
	path = path.substr(0, path.find_last_of('/') + 1);
	path += "//Lovense-browser-page";
#ifdef _WIN32
	path += ".exe";
	CefMainArgs args;
#else
	/* On non-windows platforms, ie macOS, we'll want to pass thru flags to CEF */
	struct obs_cmdline_args cmdline_args = obs_get_cmdline_args();
	CefMainArgs args(cmdline_args.argc, cmdline_args.argv);
#endif

	CefSettings settings;
	settings.log_severity = LOGSEVERITY_DISABLE;
	settings.windowless_rendering_enabled = true;
	settings.no_sandbox = true;
#if defined(__APPLE__) && !defined(BROWSER_DEPLOY)
	CefString(&settings.framework_dir_path) = CEF_LIBRARY;
#endif

	std::string obs_locale = obs_get_locale();
	std::string accepted_languages;
	if (obs_locale != "en-US") {
		accepted_languages = obs_locale;
		accepted_languages += ",";
		accepted_languages += "en-US,en";
	}
	else {
		accepted_languages = "en-US,en";
	}

	BPtr<char> conf_path = obs_module_config_path("");
	os_mkdir(conf_path);
	BPtr<char> conf_path_abs = os_get_abs_path_ptr(conf_path);
	CefString(&settings.locale) = obs_get_locale();
	CefString(&settings.accept_language_list) = accepted_languages;
	CefString(&settings.cache_path) = conf_path_abs;
	CefString(&settings.browser_subprocess_path) = path;

	bool tex_sharing_avail = false;

#if EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
	if (hwaccel) {
		obs_enter_graphics();
		hwaccel = tex_sharing_avail = gs_shared_texture_available();
		obs_leave_graphics();
	}
#endif

	CefRefPtr<BrowserApp> app(new BrowserApp(tex_sharing_avail));
	CefExecuteProcess(args, app, nullptr);
	CefInitialize(args, settings, app, nullptr);
	CefRegisterSchemeHandlerFactory("http", "absolute",
		new BrowserSchemeHandlerFactory());
	os_event_signal(cef_started_event);


	CefRunMessageLoop();
	CefShutdown();
}

extern "C" EXPORT void obs_browser_initialize(void)
{
	static bool manager_initialized = false;
	if (!os_atomic_set_bool(&manager_initialized, true)) {
		manager_thread = thread(BrowserManagerThread);
	}
}

void RegisterBrowserSource()
{
	struct obs_source_info info = {};
	info.id = "Lovese_browser_source";
	info.type = OBS_SOURCE_TYPE_INPUT;
	info.output_flags = OBS_SOURCE_VIDEO |
		OBS_SOURCE_CUSTOM_DRAW |
		OBS_SOURCE_INTERACTION |
		OBS_SOURCE_DO_NOT_DUPLICATE;
	info.get_properties = browser_source_get_properties;
	info.get_defaults = browser_source_get_defaults;

	info.get_name = [](void *)
	{
		return obs_module_text("Lovense Video Feedback");
	};
	info.create = [](obs_data_t *settings, obs_source_t *source) -> void *
	{
		obs_browser_initialize();
		BrowserSource* bs = new BrowserSource(settings, source);
		return bs;
	};
	info.destroy = [](void *data)
	{
		BrowserSource* bs = static_cast<BrowserSource *>(data);
		delete bs;
	};
	info.update = [](void *data, obs_data_t *settings)
	{
		static_cast<BrowserSource *>(data)->Update(settings);
	};
	info.get_width = [](void *data)
	{
		return (uint32_t)static_cast<BrowserSource *>(data)->width;
	};
	info.get_height = [](void *data)
	{
		return (uint32_t)static_cast<BrowserSource *>(data)->height;
	};
	info.video_tick = [](void *data, float)
	{
		static_cast<BrowserSource *>(data)->Tick();
	};
	info.video_render = [](void *data, gs_effect_t *)
	{
		static_cast<BrowserSource *>(data)->Render();
	};
	info.mouse_click = [](
		void *data,
		const struct obs_mouse_event *event,
		int32_t type,
		bool mouse_up,
		uint32_t click_count)
	{
		static_cast<BrowserSource *>(data)->SendMouseClick(
			event, type, mouse_up, click_count);
	};
	info.mouse_move = [](
		void *data,
		const struct obs_mouse_event *event,
		bool mouse_leave)
	{
		static_cast<BrowserSource *>(data)->SendMouseMove(
			event, mouse_leave);
	};
	info.mouse_wheel = [](
		void *data,
		const struct obs_mouse_event *event,
		int x_delta,
		int y_delta)
	{
		static_cast<BrowserSource *>(data)->SendMouseWheel(
			event, x_delta, y_delta);
	};
	info.focus = [](void *data, bool focus)
	{
		static_cast<BrowserSource *>(data)->SendFocus(focus);
	};
	info.key_click = [](
		void *data,
		const struct obs_key_event *event,
		bool key_up)
	{
		static_cast<BrowserSource *>(data)->SendKeyClick(event, key_up);
	};
	info.show = [](void *data)
	{
		static_cast<BrowserSource *>(data)->SetShowing(true);
	};
	info.hide = [](void *data)
	{
		static_cast<BrowserSource *>(data)->SetShowing(false);
	};
	info.activate = [](void *data)
	{
		BrowserSource *bs = static_cast<BrowserSource *>(data);
		if (bs->restart)
			bs->Refresh();
		bs->SetActive(true);
	};
	info.deactivate = [](void *data)
	{
		static_cast<BrowserSource *>(data)->SetActive(false);
	};
	obs_register_source(&info);
}

/* ========================================================================= */

extern void DispatchJSEvent(const char *eventName, const char *jsonString);

static void handle_obs_frontend_event(enum obs_frontend_event event, void *)
{
	switch (event) {
	case OBS_FRONTEND_EVENT_STREAMING_STARTING:
		DispatchJSEvent("obsStreamingStarting", nullptr);
		break;
	case OBS_FRONTEND_EVENT_STREAMING_STARTED:
		DispatchJSEvent("obsStreamingStarted", nullptr);
		break;
	case OBS_FRONTEND_EVENT_STREAMING_STOPPING:
		DispatchJSEvent("obsStreamingStopping", nullptr);
		break;
	case OBS_FRONTEND_EVENT_STREAMING_STOPPED:
		DispatchJSEvent("obsStreamingStopped", nullptr);
		break;
	case OBS_FRONTEND_EVENT_RECORDING_STARTING:
		DispatchJSEvent("obsRecordingStarting", nullptr);
		break;
	case OBS_FRONTEND_EVENT_RECORDING_STARTED:
		DispatchJSEvent("obsRecordingStarted", nullptr);
		break;
	case OBS_FRONTEND_EVENT_RECORDING_STOPPING:
		DispatchJSEvent("obsRecordingStopping", nullptr);
		break;
	case OBS_FRONTEND_EVENT_RECORDING_STOPPED:
		DispatchJSEvent("obsRecordingStopped", nullptr);
		break;
	case OBS_FRONTEND_EVENT_SCENE_CHANGED:
	{
		OBSSource source = obs_frontend_get_current_scene();
		obs_source_release(source);

		if (!source)
			break;

		const char *name = obs_source_get_name(source);
		if (!name)
			break;

		Json json = Json::object{
			{"name", name},
			{"width", (int)obs_source_get_width(source)},
			{"height", (int)obs_source_get_height(source)}
		};

		DispatchJSEvent("obsSceneChanged", json.dump().c_str());
		break;
	}
	case OBS_FRONTEND_EVENT_EXIT:
		DispatchJSEvent("obsExit", nullptr);
		break;
	default:;
	}
}

#ifdef _WIN32
static inline void EnumAdapterCount()
{
	ComPtr<IDXGIFactory1> factory;
	ComPtr<IDXGIAdapter1> adapter;
	HRESULT hr;
	UINT i = 0;

	hr = CreateDXGIFactory1(__uuidof(IDXGIFactory1), (void**)&factory);
	if (FAILED(hr))
		return;

	while (factory->EnumAdapters1(i++, &adapter) == S_OK) {
		DXGI_ADAPTER_DESC desc;

		hr = adapter->GetDesc(&desc);
		if (FAILED(hr))
			continue;

		if (i == 1)
			deviceId = desc.Description;

		/* ignore Microsoft's 'basic' renderer' */
		if (desc.VendorId == 0x1414 && desc.DeviceId == 0x8c)
			continue;

		adapterCount++;
	}
}
#endif


static wstring string2wstring(string str)
{
	wstring result;
	int len = MultiByteToWideChar(CP_ACP, 0, str.c_str(), str.size(), NULL, 0);
	TCHAR* buffer = new TCHAR[len + 1];
	MultiByteToWideChar(CP_ACP, 0, str.c_str(), str.size(), buffer, len);
	buffer[len] = '\0';            
	result.append(buffer);
	delete[] buffer;
	return result;
}

static string wstring2string(wstring wstr)
{
	string result;
	int len = WideCharToMultiByte(CP_ACP, 0, wstr.c_str(), wstr.size(), NULL, 0, NULL, NULL);
	char* buffer = new char[len + 1];
	WideCharToMultiByte(CP_ACP, 0, wstr.c_str(), wstr.size(), buffer, len, NULL, NULL);
	buffer[len] = '\0';
	result.append(buffer);
	delete[] buffer;
	return result;
}

static wstring ReadInitFile(wstring strGroun, wstring strKey)
{
	char path[512] = { 0 };
	if (os_get_config_path(path, sizeof(path), "obs-studio\\PluginVersion.ini") <= 0)
	{
		blog(LOG_ERROR, "[Lovese] ReadVersionConfig -> obs-studio\\PluginVersion.ini fail !");
		return wstring();
	}
	wstring strAppPath;
	strAppPath = string2wstring(path);
	TCHAR Result[MAX_PATH] = { 0 };
	DWORD num = GetPrivateProfileString(strGroun.c_str(), strKey.c_str(), NULL, Result, MAX_PATH, strAppPath.c_str());
	if (num == 0)
		return wstring();
	blog(LOG_ERROR, "[Lovese] GetPrivateProfileString num = %d ",num);

	wstring value = Result;
	return value;
}


#if EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
static const wchar_t *blacklisted_devices[] = {
	L"Intel",
	L"Microsoft",
	L"Radeon HD 8850M",
	L"Radeon HD 7660",
	nullptr
};

#endif



bool obs_module_load(void)
{

	blog(LOG_INFO, "[Lovese]: Version %s", OBS_BROWSER_VERSION_STRING);
	blog(LOG_INFO, "[Lovese] obs_module_load start! ");
	os_event_init(&cef_started_event, OS_EVENT_TYPE_MANUAL);
	CefEnableHighDPISupport();
#ifdef _WIN32
	EnumAdapterCount();
#endif
	RegisterBrowserSource();
	obs_frontend_add_event_callback(handle_obs_frontend_event, nullptr);

#if EXPERIMENTAL_SHARED_TEXTURE_SUPPORT_ENABLED
	obs_data_t *private_data = obs_get_private_data();
	hwaccel = obs_data_get_bool(private_data, "BrowserHWAccel");
	if (hwaccel) {
		/* do not use hardware acceleration if a blacklisted device is
		 * the default and on 2 or more adapters */
		const wchar_t **device = blacklisted_devices;
		while (*device) {
			if (!!wstrstri(deviceId.c_str(), *device)) {
				hwaccel = false;
				blog(LOG_INFO, "[Lovese]: "
					"Blacklisted device "
					"detected, "
					"disabling browser "
					"source hardware "
					"acceleration.");
				break;
			}
			device++;
		}
	}
	obs_data_release(private_data);
#endif

	char path[512] = { 0 };
	if (os_get_config_path(path, sizeof(path), "obs-studio/dist") <= 0)
		blog(LOG_ERROR, "[Lovese] os_get_config_path locale dist path is  fail ");
	if (!os_mkdir(path))
		blog(LOG_ERROR, "[Lovese] os_mkdir locale dist path is  fail ");

	blog(LOG_INFO, "[Lovese] websocket_server_start ! ");
	websocket_server_start(81000);
	return true;
}

void obs_module_unload(void)
{
	blog(LOG_INFO, "[Lovese] obs_module_unload start! ");
	websocket_server_stop();

	if (manager_thread.joinable()) {
		while (!QueueCEFTask([]() {CefQuitMessageLoop(); }))
			os_sleep_ms(5);

		manager_thread.join();
	}

	if (cef_started_event)
	{
		os_event_destroy(cef_started_event);
	}
	blog(LOG_INFO, "[Lovese] obs_module_unload end! ");
}

#include <fstream>
#include <sstream>
#include "CJsonObject.hpp"
bool get_lovense_source_config(neb::CJsonObject & pJson)
{
	const char * json_buff = "{\
			\"balance\": 0.5,\
			\"deinterlace_field_order\" : 0,\
			\"deinterlace_mode\" : 0,\
			\"enabled\" : true,\
			\"flags\" : 0,\
			\"hotkeys\" : {},\
			\"id\" : \"Lovese_browser_source\",\
			\"mixers\" : 0,\
			\"monitoring_type\" : 0,\
			\"muted\" : false,\
			\"name\" : \"Lovense Video Feedback\",\
			\"private_settings\" : {},\
			\"push-to-mute\" : false,\
			\"push-to-mute-delay\" : 0,\
			\"push-to-talk\" : false,\
			\"push-to-talk-delay\" : 0,\
			\"settings\" : {\
				\"fps_custom\": true\
				},\
			\"sync\" : 0,\
			\"volume\" : 1.0\
		}";
	return pJson.Parse(json_buff);
}

bool get_lovense_setting_items_config(neb::CJsonObject & pJson)
{
	const char * json_buff = "{\"align\": 5,\
			\"bounds\": {\
			\"x\": 0.0,\
			\"y\" : 0.0},\
			\"bounds_align\" : 0,\
			\"bounds_type\" : 0,\
			\"crop_bottom\" : 0,\
			\"crop_left\" : 0,\
			\"crop_right\" : 0,\
			\"crop_top\" : 0,\
			\"group_item_backup\" : false,\
			\"id\" : 150,\
			\"locked\" : false,\
			\"name\" : \"Lovense Video Feedback\",\
			\"pos\" : {\
				\"x\": 0.0,\
				\"y\" : 0.0},\
			\"private_settings\" : {},\
				\"rot\" : 0.0,\
				\"scale\" : {\
					\"x\": 1.0,\
					\"y\" : 1.0\
				},\
			\"scale_filter\" : \"disable\",\
			\"visible\" : true}";

	return pJson.Parse(json_buff);
}

bool lovense_plugins_is_setup()
{
	char path[512] = { 0 };
	if (os_get_config_path(path, sizeof(path), "obs-studio\\PluginVersion.ini") <= 0)
	{
		blog(LOG_ERROR, "[Lovese] ReadVersionConfig -> obs-studio\\PluginVersion.ini fail !");
		//ļڣ
		return true;
	}
	wstring strAppPath = string2wstring(path);
	TCHAR Result[MAX_PATH] = { 0 };
	GetPrivateProfileString(L"OBS_PLUGIN", L"LovenseInit", L"0", Result, MAX_PATH, strAppPath.c_str());
	wstring wstrResult = Result;
	if (wstrResult == L"0")
	{
		return false;
	}
	else
		return true;
}

void setup_lovense_plugin_config()
{
	char path[512] = { 0 };
	{
		blog(LOG_ERROR, "[Lovese] WriteVersionConfig -> obs-studio\\PluginVersion.ini fail !");
	}
	wstring  strAppPath = string2wstring(path);
	WritePrivateProfileString(L"OBS_PLUGIN", L"LovenseInit", L"1", strAppPath.c_str());
}

void add_sense_to_obs()
{
	if (lovense_plugins_is_setup())
		return;

	char szobs_config_path[MAX_PATH] = { 0 };
	if (os_get_config_path(szobs_config_path, MAX_PATH, "obs-studio\\basic\\scenes") <= 0)
	{
		blog(LOG_INFO, "[Lovense] obs sense path is Not exit!");
		return;
	}
	setup_lovense_plugin_config();

	wstring wstr_sense_path = string2wstring(szobs_config_path);
	strcat_s(szobs_config_path, "\\*.json");

	WIN32_FIND_DATA data;
	HANDLE pHandle = NULL;
	wchar_t wcstr_path;
	wstring wstrPath = string2wstring(szobs_config_path);
	pHandle = FindFirstFile(wstrPath.c_str(), &data);
	if (pHandle)
	{
		wstr_sense_path += wstring(L"\\");
		wstr_sense_path += data.cFileName;
		if (PathFileExists(wstr_sense_path.c_str()))
		{
			string strPath = wstring2string(wstr_sense_path);
			ifstream pFile(strPath.c_str(), ios::in);
			char file_buff[1024] = { 0 };
			string strJson;
			if (pFile.is_open())
			{
				while (pFile.getline(file_buff, 1024))
				{
					strJson += file_buff;
					memset(file_buff, 0, sizeof(file_buff));
				}
				pFile.close();

				if (strJson.find("Lovense Video Feedback") != strJson.npos)
					return;

				neb::CJsonObject pRootJson;
				if (pRootJson.Parse(strJson))
				{
					neb::CJsonObject scene_order = pRootJson["scene_order"];
					std::string first_scene_name;
					if (!scene_order.IsEmpty())
					{
						if (scene_order.IsArray() && scene_order.GetArraySize() > 0)
						{
							neb::CJsonObject pItem = scene_order[0];
							if (!pItem.IsEmpty())
							{
								first_scene_name = pItem("name");
							}
						}
					}

					neb::CJsonObject sources = pRootJson["sources"];
					if (!sources.IsEmpty())
					{
						if (sources.IsArray())
						{
							int arraySize = sources.GetArraySize();
							for (int i = 0; i < arraySize; i++)
							{
								neb::CJsonObject oneItem = sources[i];
								if (oneItem("name") == first_scene_name)
								{
									neb::CJsonObject lovesen_source_item;
									if (get_lovense_setting_items_config(lovesen_source_item))
										pRootJson["sources"][i]["settings"]["items"].Add(lovesen_source_item);
									break;
								}
							}

							neb::CJsonObject lovense_source_item;
							if (get_lovense_source_config(lovense_source_item))
							{
								pRootJson["sources"].Add(lovense_source_item);
								string dump = pRootJson.ToFormattedString();
								ofstream outFile;
								outFile.open(strPath, ios_base::out | ios_base::trunc);
								if (outFile.is_open())
								{
									stringstream ostrStream;
									ostrStream << dump;
									outFile << ostrStream.str() << endl;
									outFile.flush();
									outFile.close();
								}
							}
						}
					}
				}

			}
		}
	}
}


